/*
 * Decompiled with CFR 0.152.
 */
package loci.formats.out;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import loci.common.RandomAccessInputStream;
import loci.common.RandomAccessOutputStream;
import loci.formats.FormatException;
import loci.formats.FormatTools;
import loci.formats.FormatWriter;
import loci.formats.MetadataTools;
import loci.formats.meta.MetadataRetrieve;
import ome.units.UNITS;
import ome.units.quantity.Time;

public class ICSWriter
extends FormatWriter {
    private long dimensionOffset;
    private int dimensionLength;
    private long pixelOffset;
    private int lastPlane = -1;
    private RandomAccessOutputStream pixels;
    private List<String> uniqueFiles = new ArrayList<String>();
    private String outputOrder = "XYZTC";

    public ICSWriter() {
        super("Image Cytometry Standard", new String[]{"ids", "ics"});
    }

    public void setOutputOrder(String outputOrder) {
        this.outputOrder = outputOrder;
    }

    @Override
    public void saveBytes(int no, byte[] buf, int x, int y, int w, int h) throws FormatException, IOException {
        this.checkParams(no, buf, x, y, w, h);
        if (this.pixels == null) {
            this.pixels = new RandomAccessOutputStream(this.currentId);
        }
        MetadataRetrieve meta = this.getMetadataRetrieve();
        int rgbChannels = this.getSamplesPerPixel();
        String order = meta.getPixelsDimensionOrder(this.series).getValue();
        int sizeZ = (Integer)meta.getPixelsSizeZ(this.series).getValue();
        int sizeC = meta.getChannelCount(this.series);
        if (rgbChannels <= sizeC) {
            sizeC /= rgbChannels;
        }
        int sizeT = (Integer)meta.getPixelsSizeT(this.series).getValue();
        int planes = sizeZ * sizeC * sizeT;
        int[] coords = FormatTools.getZCTCoords(order, sizeZ, sizeC, sizeT, planes, no);
        int realIndex = FormatTools.getIndex(this.outputOrder, sizeZ, sizeC, sizeT, planes, coords[0], coords[1], coords[2]);
        if (this.uniqueFiles.size() > 1) {
            realIndex = no;
        }
        int sizeX = (Integer)meta.getPixelsSizeX(this.series).getValue();
        int sizeY = (Integer)meta.getPixelsSizeY(this.series).getValue();
        int pixelType = FormatTools.pixelTypeFromString(meta.getPixelsType(this.series).toString());
        int bytesPerPixel = FormatTools.getBytesPerPixel(pixelType);
        long planeSize = sizeX * sizeY * rgbChannels * bytesPerPixel;
        if (!this.initialized[this.series][realIndex]) {
            this.initialized[this.series][realIndex] = true;
            if (!this.isFullPlane(x, y, w, h)) {
                this.pixels.seek(this.pixelOffset + (long)(realIndex + 1) * planeSize);
            }
        }
        this.pixels.seek(this.pixelOffset + (long)realIndex * planeSize);
        if (this.isFullPlane(x, y, w, h) && (this.interleaved || rgbChannels == 1)) {
            this.pixels.write(buf);
        } else {
            this.pixels.skipBytes(bytesPerPixel * rgbChannels * sizeX * y);
            for (int row = 0; row < h; ++row) {
                ByteArrayOutputStream strip = new ByteArrayOutputStream();
                for (int col = 0; col < w; ++col) {
                    for (int c = 0; c < rgbChannels; ++c) {
                        int index = this.interleaved ? rgbChannels * (row * w + col) + c : w * (c * h + row) + col;
                        strip.write(buf, index * bytesPerPixel, bytesPerPixel);
                    }
                }
                this.pixels.skipBytes(bytesPerPixel * rgbChannels * x);
                this.pixels.write(strip.toByteArray());
                this.pixels.skipBytes(bytesPerPixel * rgbChannels * (sizeX - w - x));
            }
        }
        this.lastPlane = realIndex;
        this.overwriteDimensions(this.getMetadataRetrieve());
        this.pixels.close();
        this.pixels = null;
    }

    @Override
    public boolean canDoStacks() {
        return true;
    }

    @Override
    public int[] getPixelTypes(String codec) {
        return new int[]{0, 1, 2, 3, 4, 5, 6};
    }

    @Override
    public void setId(String id) throws FormatException, IOException {
        super.setId(id);
        if (!this.uniqueFiles.contains(id)) {
            this.uniqueFiles.add(id);
        }
        if (ICSWriter.checkSuffix(this.currentId, "ids")) {
            String metadataFile = this.currentId.substring(0, this.currentId.lastIndexOf("."));
            metadataFile = metadataFile + ".ics";
            this.out.close();
            this.out = new RandomAccessOutputStream(metadataFile);
        }
        if (this.out.length() == 0L) {
            this.out.writeBytes("\t\n");
            if (ICSWriter.checkSuffix(id, "ids")) {
                this.out.writeBytes("ics_version\t1.0\n");
            } else {
                this.out.writeBytes("ics_version\t2.0\n");
            }
            this.out.writeBytes("filename\t" + this.currentId + "\n");
            this.out.writeBytes("layout\tparameters\t6\n");
            MetadataRetrieve meta = this.getMetadataRetrieve();
            MetadataTools.verifyMinimumPopulated(meta, this.series);
            int pixelType = FormatTools.pixelTypeFromString(meta.getPixelsType(this.series).toString());
            this.dimensionOffset = this.out.getFilePointer();
            int[] sizes = this.overwriteDimensions(meta);
            this.dimensionLength = (int)(this.out.getFilePointer() - this.dimensionOffset);
            if (this.validBits != 0) {
                this.out.writeBytes("layout\tsignificant_bits\t" + this.validBits + "\n");
            }
            boolean signed = FormatTools.isSigned(pixelType);
            boolean littleEndian = false;
            if (meta.getPixelsBigEndian(this.series) != null) {
                littleEndian = meta.getPixelsBigEndian(this.series) == false;
            } else if (meta.getPixelsBinDataCount(this.series) == 0) {
                littleEndian = meta.getPixelsBinDataBigEndian(this.series, 0) == false;
            }
            this.out.writeBytes("representation\tformat\t" + (pixelType == 6 ? "real\n" : "integer\n"));
            this.out.writeBytes("representation\tsign\t" + (signed ? "signed\n" : "unsigned\n"));
            this.out.writeBytes("representation\tcompression\tuncompressed\n");
            this.out.writeBytes("representation\tbyte_order\t");
            for (int i = 0; i < sizes[0] / 8; ++i) {
                if (littleEndian && (sizes[0] < 32 || pixelType == 6) || !littleEndian && sizes[0] >= 32 && pixelType != 6) {
                    this.out.writeBytes(i + 1 + "\t");
                    continue;
                }
                this.out.writeBytes(sizes[0] / 8 - i + "\t");
            }
            this.out.writeBytes("\nparameter\tscale\t1.000000\t");
            StringBuilder units = new StringBuilder();
            for (int i = 0; i < this.outputOrder.length(); ++i) {
                Time valueTime;
                char dim = this.outputOrder.charAt(i);
                Number value = 1.0;
                if (dim == 'X') {
                    if (meta.getPixelsPhysicalSizeX(0) != null) {
                        value = meta.getPixelsPhysicalSizeX(0).value(UNITS.MICROMETER).doubleValue();
                    }
                    units.append("micrometers\t");
                } else if (dim == 'Y') {
                    if (meta.getPixelsPhysicalSizeY(0) != null) {
                        value = meta.getPixelsPhysicalSizeY(0).value(UNITS.MICROMETER).doubleValue();
                    }
                    units.append("micrometers\t");
                } else if (dim == 'Z') {
                    if (meta.getPixelsPhysicalSizeZ(0) != null) {
                        value = meta.getPixelsPhysicalSizeZ(0).value(UNITS.MICROMETER).doubleValue();
                    }
                    units.append("micrometers\t");
                } else if (dim == 'T' && (valueTime = meta.getPixelsTimeIncrement(0)) != null) {
                    value = valueTime.value(UNITS.SECOND);
                    units.append("seconds\t");
                }
                this.out.writeBytes(value + "\t");
            }
            this.out.writeBytes("\nparameter\tunits\tbits\t" + units.toString() + "\n");
            this.out.writeBytes("\nend\n");
            this.pixelOffset = this.out.getFilePointer();
        } else if (ICSWriter.checkSuffix(this.currentId, "ics")) {
            try (RandomAccessInputStream in = new RandomAccessInputStream(this.currentId);){
                in.findString("\nend\n");
                this.pixelOffset = in.getFilePointer();
            }
        }
        if (ICSWriter.checkSuffix(this.currentId, "ids")) {
            this.pixelOffset = 0L;
        }
    }

    @Override
    public void close() throws IOException {
        super.close();
        this.pixelOffset = 0L;
        this.lastPlane = -1;
        this.dimensionOffset = 0L;
        this.dimensionLength = 0;
        if (this.pixels != null) {
            this.pixels.close();
        }
        this.pixels = null;
        this.uniqueFiles.clear();
    }

    private int[] overwriteDimensions(MetadataRetrieve meta) throws IOException {
        int i;
        this.out.seek(this.dimensionOffset);
        int sizeX = (Integer)meta.getPixelsSizeX(this.series).getValue();
        int sizeY = (Integer)meta.getPixelsSizeY(this.series).getValue();
        int z = (Integer)meta.getPixelsSizeZ(this.series).getValue();
        int c = (Integer)meta.getPixelsSizeC(this.series).getValue();
        int t = (Integer)meta.getPixelsSizeT(this.series).getValue();
        int pixelType = FormatTools.pixelTypeFromString(meta.getPixelsType(this.series).toString());
        int bytesPerPixel = FormatTools.getBytesPerPixel(pixelType);
        int rgbChannels = this.getSamplesPerPixel();
        if (this.lastPlane < 0) {
            this.lastPlane = z * c * t - 1;
        }
        int[] pos = FormatTools.getZCTCoords(this.outputOrder, z, c, t, z * c * t, this.lastPlane);
        StringBuilder dimOrder = new StringBuilder();
        int[] sizes = new int[6];
        int nextSize = 0;
        sizes[nextSize++] = 8 * bytesPerPixel;
        if (rgbChannels > 1) {
            dimOrder.append("ch\t");
            sizes[nextSize++] = rgbChannels;
        }
        for (i = 0; i < this.outputOrder.length(); ++i) {
            if (this.outputOrder.charAt(i) == 'X') {
                sizes[nextSize++] = sizeX;
            } else if (this.outputOrder.charAt(i) == 'Y') {
                sizes[nextSize++] = sizeY;
            } else if (this.outputOrder.charAt(i) == 'Z') {
                sizes[nextSize++] = pos[0] + 1;
            } else if (this.outputOrder.charAt(i) == 'T') {
                sizes[nextSize++] = pos[2] + 1;
            } else if (this.outputOrder.charAt(i) == 'C' && dimOrder.indexOf("ch") == -1) {
                sizes[nextSize++] = pos[1] + 1;
                dimOrder.append("ch");
            }
            if (this.outputOrder.charAt(i) != 'C') {
                dimOrder.append(String.valueOf(this.outputOrder.charAt(i)).toLowerCase());
            }
            dimOrder.append("\t");
        }
        this.out.writeBytes("layout\torder\tbits\t" + dimOrder.toString() + "\n");
        this.out.writeBytes("layout\tsizes\t");
        for (i = 0; i < sizes.length; ++i) {
            this.out.writeBytes(sizes[i] + "\t");
        }
        while (this.out.getFilePointer() - this.dimensionOffset < (long)(this.dimensionLength - 1)) {
            this.out.writeBytes(" ");
        }
        this.out.writeBytes("\n");
        return sizes;
    }
}

